#!/bin/bash

#/*******************************************************************
# * This file is part of the Emulex Linux Device Driver for         *
# * Fibre Channel Host Bus Adapters.                                *
# * Copyright (C) 2004-2006 Emulex.  All rights reserved.           *
# * EMULEX and SLI are trademarks of Emulex.                        *
# * www.emulex.com                                                  *
# *                                                                 *
# * This program is free software; you can redistribute it and/or   *
# * modify it under the terms of version 2 of the GNU General       *
# * Public License as published by the Free Software Foundation.    *
# * This program is distributed in the hope that it will be useful. *
# * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND          *
# * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,  *
# * FITNESS FOR A PARTICULAR PURPOSE, OR NON-INFRINGEMENT, ARE      *
# * DISCLAIMED, EXCEPT TO THE EXTENT THAT SUCH DISCLAIMERS ARE HELD *
# * TO BE LEGALLY INVALID.  See the GNU General Public License for  *
# * more details, a copy of which can be found in the file COPYING  *
# * included with this package.                                     *
# *******************************************************************/
#
###########################################################################
#
#			    elxreport.sh
#	Shell program that gathers system, HBA and device driver/applications
#       information related to the functionality of Emulex linux device 
#       drivers.
#       The intented targets are Emulex customers with installed linux device 
#       drivers.
#       This tool creates a tarball file which we ask the customers to send 
#       to Emulex support for analysis.
#
#	Usage:
#		elxreport.sh [ -h | --help ]
#
#	Options:
#		-h, --help	Display this help message and exit.
#
#	Revisions:
#	1.0	11/01/2004	Vaios Papadimitriou, Jerry Lindsey   Genesis
#
###########################################################################

###########################################################################
# Constants
###########################################################################
PROGNAME=$(basename $0)
VERSION="1.4"

DATE=$(/bin/date +%m%d%y%H%M%S)
TOPDIR=/tmp/${PROGNAME}
TMPDIR=${TOPDIR}/${PROGNAME}_${DATE}
TMPLOG=${TMPDIR}/${PROGNAME}.log
OS_REV=`uname -r | cut -c1-3`

###########################################################################
# Functions
###########################################################################

#
# Function to display usage message.
#
function usage()
{
	echo "Usage: ${PROGNAME} [-h | --help]"
	echo
	echo "Shell program that gathers system, HBA and device driver/applications"
        echo "information related to the functionality of Emulex linux device drivers."
	echo "This tool creates a tarball file /tmp/${PROGNAME}_<date>.tgz to be send" 
	echo "to Emulex support for analysis."
	echo
}

#
# Function to execute a shell command.
#
exec_cmd()
{
	echo "" >> $TMPLOG
	echo "Command: ${1} " >> $TMPLOG
	eval ${1}  >> $TMPLOG 2>&1
	echo "" >> $TMPLOG 
}

#
# Function to copy data to the log file.
#
exec_copy()
{
	echo "" >> $TMPLOG
	echo "Copying output from: ${1}" >> $TMPLOG
	
	if [ -f ${1} ] ; then 
		cp --parents ${1} $TMPDIR  >> $TMPLOG
	fi
	echo "" >> $TMPLOG
}

#
# Function to create temporary directories and files and display banner
# information.
#
init_files()
{
	# Must run as root
	if [ $UID != 0 ]; then
		echo "You must be logged on as root."
		exit 1
	fi

	# Create temp dir to store results
	rm -rf $TOPDIR
	mkdir -p $TMPDIR

	> $TMPLOG

	echo ""
	echo "Emulex Corporation - ELX Report Utility, version ${VERSION}" | tee -a $TMPLOG
	echo "Date:$(date)" | tee -a $TMPLOG

	echo "Initializing report environment for host:${HOSTNAME}"
}

#
# Function to clean-up temporary directories and files.
#
clean_up()
{
	rm -rf ${TOPDIR}
}

# instr p1 p2 - return 1 if p2 exists in p1 
instr()
{

let FOUND=0
for j in $2; do
   if [ $1 = $j ] ; then
      let FOUND=1
   fi
done

return $FOUND

}

# process_iosched C_DIR D_DIR parm
process_iosched()
{
local C_DIR
local D_DIR
local parm
local readok
local EXCLUDELIST
local RETURNVALUE

C_DIR="$1/$3"
D_DIR="$2-$3"

for parm in `ls $C_DIR` 
  do
        if  [ -d $C_DIR/$parm ] ; then 
            echo "$D_DIR: Unexpected directory found:$C_DIR/$parm"
        else 
    	    readok=`ls -l $C_DIR/$parm | cut -c2-2`
	    if  [ "$readok" = "r" ]  &&  [ ! -L $C_DIR/$parm ] && [ ! -d $C_DIR/$parm ] ; then 
		instr "$parm" "$EXCLUDELIST"
		RETURNVALUE=$?
		if [ ${RETURNVALUE} -eq 0 ]; then
		    echo "$D_DIR: $parm="`cat $C_DIR/$parm` >> $TMPLOG
		fi
            fi    
        fi
done
      
return

}
# process_iosched C_DIR D_DIR parm
process_queue()
{
local C_DIR
local D_DIR
local parm
local readok
local EXCLUDELIST
local RETURNVALUE

C_DIR="$1/$3"
D_DIR="$2-$3"

for parm in `ls $C_DIR` 
  do
        if  [ -d $C_DIR/$parm ] ; then 
            if  [ "$parm" = "iosched" ]; then 
                process_iosched $C_DIR $D_DIR $parm
            fi
        else 
    	    readok=`ls -l $C_DIR/$parm | cut -c2-2`
	    if  [ "$readok" = "r" ]  &&  [ ! -L $C_DIR/$parm ] && [ ! -d $C_DIR/$parm ] ; then 
		instr "$parm" "$EXCLUDELIST"
		RETURNVALUE=$?
		if [ ${RETURNVALUE} -eq 0 ]; then
		    echo "$D_DIR: $parm="`cat $C_DIR/$parm` >> $TMPLOG
		fi
            fi    
        fi
done
      
return

}


# process_block C_DIR D_DIR parm
process_block()
{
local C_DIR
local D_DIR
local parm
local readok
local EXCLUDELIST
local RETURNVALUE

C_DIR="$1/$3"
D_DIR="$2-$3"

for parm in `ls $C_DIR` 
  do
        if  [ -d $C_DIR/$parm ] ; then 
            if  [ "$parm" = "queue" ]; then 
                process_queue $C_DIR $D_DIR $parm
            fi
        else 
    	    readok=`ls -l $C_DIR/$parm | cut -c2-2`
	    if  [ "$readok" = "r" ]  &&  [ ! -L $C_DIR/$parm ] && [ ! -d $C_DIR/$parm ] ; then 
		instr "$parm" "$EXCLUDELIST"
		RETURNVALUE=$?
		if [ ${RETURNVALUE} -eq 0 ]; then
		    echo "$D_DIR: $parm="`cat $C_DIR/$parm` >> $TMPLOG
		fi
            fi    
        fi
done
      
return

}

# process_lun C_DIR D_DIR parm
process_lun()
{
local C_DIR
local D_DIR
local parm
local readok
local EXCLUDELIST
local RETURNVALUE

C_DIR="$1/$3"
D_DIR="$2-$3"

for parm in `ls $C_DIR` 
  do
        if  [ -d $C_DIR/$parm ] ; then 
            if  [ "$parm" = "block" ]; then 
                echo "$D_DIR: "`ls -l $C_DIR/$parm` >> $TMPLOG
                process_block $C_DIR $D_DIR $parm
            fi
        else 
    	    readok=`ls -l $C_DIR/$parm | cut -c2-2`
	    if  [ "$readok" = "r" ]  &&  [ ! -L $C_DIR/$parm ] && [ ! -d $C_DIR/$parm ] ; then 
		instr "$parm" "$EXCLUDELIST"
		RETURNVALUE=$?
		if [ ${RETURNVALUE} -eq 0 ]; then
		    echo "$D_DIR: $parm="`cat $C_DIR/$parm` >> $TMPLOG
		fi
            fi    
        fi
done
      
return

}
     
# process_target C_DIR D_DIR parm
process_target()
{
local C_DIR
local D_DIR
local parm
local readok
local EXCLUDELIST
local RETURNVALUE

C_DIR="$1/$3"
D_DIR="$2-$3"

for parm in `ls $C_DIR` 
  do
        if  [ -d $C_DIR/$parm ] ; then 
            process_lun $C_DIR $D_DIR $parm
        else 
    	    readok=`ls -l $C_DIR/$parm | cut -c2-2`
	    if  [ "$readok" = "r" ]  &&  [ ! -L $C_DIR/$parm ] && [ ! -d $C_DIR/$parm ] ; then 
		instr "$parm" "$EXCLUDELIST"
		RETURNVALUE=$?
		if [ ${RETURNVALUE} -eq 0 ]; then
		    echo "$D_DIR: $parm="`cat $C_DIR/$parm` >> $TMPLOG
		fi
            fi    
        fi
done
      
return

}


# process_device C_DIR D_DIR parm
process_device()
{
local C_DIR
local D_DIR
local parm
local readok
local EXCLUDELIST
local RETURNVALUE

C_DIR="$1/$3"
D_DIR="$2-$3"

for parm in `ls $C_DIR` 
  do
        if  [ -d $C_DIR/$parm ] ; then 
            process_target $C_DIR $D_DIR $parm
        else 
	    readok=`ls -l $C_DIR/$parm | cut -c2-2`
	    if  [ "$readok" = "r" ]  &&  [ ! -L $C_DIR/$parm ] && [ ! -d $C_DIR/$parm ] ; then 
		instr "$parm" "$EXCLUDELIST"
		RETURNVALUE=$?
		if [ ${RETURNVALUE} -eq 0 ]; then
		    echo "$D_DIR: $parm="`cat $C_DIR/$parm` >> $TMPLOG
		fi
	    fi
        fi
done
      
return

}

dump_sys_class()
{
local f
local HOST_ID
local D_DIR
local parm
local EXCLUDELIST

EXCLUDELIST="mbox ctlreg slimem ctpass"

for f in /sys/class/scsi_host/*
  do
    if [ -f $f/lpfc_drvr_version ]; then
	   HOST_ID=`echo $f | sed -e "s/.*host//"`
           echo "" >> $TMPLOG
           echo "Processing /sys/class/scsi_host/host$HOST_ID:" >> $TMPLOG
           D_DIR="host$HOST_ID"
	   for parm in `ls $f` 
	   do
		if [ $parm = "device" ]; then
		   process_device $f $D_DIR $parm 
		fi
		readok=`ls -l $f/$parm | cut -c2-2`
		if  [ $readok = r ]  &&  [ ! -L $f/$parm ] && [ ! -d $f/$parm ] ; then 
			instr "$parm" "$EXCLUDELIST"
			RETURNVALUE=$?
			if [ ${RETURNVALUE} -eq 0 ]; then
			 echo "$D_DIR: $parm="`cat $f/$parm` >> $TMPLOG
			fi
		fi
	   done 
    fi
done

echo "" >> $TMPLOG
}
            
#
# Function to gather OS/distribution related information.
#
get_os_info()
{
	exec_cmd "/bin/uname -a"
	if [ -f /etc/redhat-release ] ; then
		exec_cmd  "cat /etc/redhat-release"
		exec_copy "/etc/redhat-release"
	elif [ -f /etc/SuSE-release ] ; then
		exec_cmd  "cat /etc/SuSE-release"
		exec_copy "/etc/SuSE-release"
	fi

        echo "OS Rev is "${OS_REV}

}

#
# Function to gather system information related to the functionality
# of the linux device driver.
#
get_system_info()
{
	exec_cmd  "hostname"
	exec_cmd  "uptime " 

	exec_copy "/proc/partitions"
	exec_copy "/proc/cpuinfo"
	exec_copy "/proc/devices"
	exec_copy "/proc/cmdline"
	exec_copy "/proc/scsi/scsi"

	exec_cmd  "gcc -v"
	exec_cmd  "/sbin/lsmod"
}

#
# Function to gather system memory information.
#
get_memory_info()
{
	exec_cmd  "vmstat"
	exec_cmd  "free"
	exec_copy "/proc/slabinfo"
}

#
# Function to gather information about installed RPMs.
#
get_rpm_info()
{
	exec_cmd "rpm -qa | sort"
}

get_pci_info()
{
	exec_cmd  "/sbin/lspci -v" 
	exec_cmd  "/sbin/lspci -n" 
	exec_cmd  "/sbin/lspci -nv" 
}

get_kernel_info()
{
	exec_cmd  "ls -la /usr/src"
	if [ -f /usr/src/linux/.config ]; then
	    exec_cmd  "cp /usr/src/linux/.config ${TMPDIR}/${KVER}.config"
	fi
}

get_message_files()
{
	exec_copy "/var/log/messages "
#	exec_copy "/var/log/messages.%"
	exec_copy "/var/log/messages.1"
	exec_copy "/var/log/messages.2"
	exec_copy "/var/log/messages.3"
	if [ -e /var/log/warn ] ; then
	    exec_copy "/var/log/warn"
	    exec_copy "/var/log/warn.1"
	    exec_copy "/var/log/warn.2"
	    exec_copy "/var/log/warn.3"
	fi
	exec_copy "/var/log/dmesg"
}

get_boot_info()
{
	exec_copy "/etc/lilo.conf"
	exec_copy "/etc/grub.conf"
	exec_copy "/boot/etc/yaboot.conf"
}

#
# Function to gather lpfc device driver information.
#
get_lpfc_info()
{
	exec_copy "/proc/scsi/scsi"
	exec_cmd  "ls /proc/scsi/lpfc"

#	exec_copy '/proc/scsi/lpfc/\*'
	if [ -d /proc/scsi/lpfc ]; then
		echo "Copying output from: /proc/scsi/lpfc/*" >> $TMPLOG
		echo "" >> $TMPLOG
		cp --parents /proc/scsi/lpfc/* $TMPDIR  >> $TMPLOG
	fi

#	exec_cmd  "find /lib/modules -name lpfc\* -exec ls -l \{\} \;"
	echo "Command: 	find /lib/modules -name lpfc\* -exec ls -l {}" >> $TMPLOG
	find /lib/modules -name lpfc\* -exec ls -l {} \; >> $TMPLOG

	echo "" >> $TMPLOG

	# Any mention of Emulex driver in  /etc/... 
	for f in $(grep -l -r -s lpfc /etc/*) ; do echo "lpfc found in:$f*" >> $TMPLOG ; cp --parents $f ${TMPDIR}; done

	echo "" >> $TMPLOG
}

#
# Function to gather Multipule driver information if available.
#
get_mp_info()
{
	if [ -d /proc/scsi/lpfcmpl ]; then
		echo "Getting Multipulse info from /proc/scsi/lpfcmpl/*..." | tee -a $TMPLOG
		echo "" >> $TMPLOG
		cp --parents /proc/scsi/lpfcmpl/* $TMPDIR  >> $TMPLOG
	fi
}


get_library_info()
{
	exec_cmd  "file /usr/lib/libdfc*.*"
	exec_cmd  "ls -la /usr/lib/libdfc*.*"
	exec_cmd  "file /usr/lib/libHBA*.*"
	exec_cmd  "ls -la /usr/lib/libHBA*.*"
	exec_cmd  "file /usr/lib/libemulexhbaapi*.*"
	exec_cmd  "ls -la /usr/lib/libemulexhbaapi*.*"
	exec_cmd  "cat /etc/hba.conf"
}


###########################################################################
#	Program starts here
###########################################################################

if [ "$1" = "--help" -o "$1" = "-h" ]; then
	usage
	exit 0
fi

# Create temp files
init_files

echo "Getting OS Info..." | tee -a $TMPLOG
get_os_info

echo "Getting System Info..." | tee -a $TMPLOG
get_system_info

echo "Checking System memory..." | tee -a $TMPLOG
get_memory_info

echo "Getting RPM Info..." | tee -a $TMPLOG
get_rpm_info

echo "Getting PCI Info..." | tee -a $TMPLOG
get_pci_info

echo "Checking kernel info..." | tee -a $TMPLOG
get_kernel_info

echo "Gathering message files..." | tee -a $TMPLOG
get_message_files

echo "Gathering boot info..." | tee -a $TMPLOG
get_boot_info

echo "Getting LPFC info..." | tee -a $TMPLOG
get_lpfc_info

if [ ${OS_REV} = 2.6 ]; then
   echo "Gathering LPFC info from /sys/class/scsi_host" | tee -a $TMPLOG
   dump_sys_class 
fi

# Copy Muiltipulse driver data if available
get_mp_info

echo "Getting library info..." | tee -a $TMPLOG
get_library_info

find ${TMPDIR} -name "*" |xargs chmod 755

echo "Generating report..." | tee -a $TMPLOG
tar -C "/tmp"  -zcvf "/tmp/${PROGNAME}_${DATE}.tgz" ${PROGNAME}/ >> $TMPLOG

echo ""
echo "Please send file /tmp/${PROGNAME}_${DATE}.tgz to your Emulex Support Representative."
echo ""

clean_up

exit 0

